<?php
if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;
$table_name = $wpdb->prefix . 'jbm_backups';
// SECURITY: Use sprintf with esc_sql for safe table name construction
$backups = $wpdb->get_results(
    sprintf(
        "SELECT * FROM `%s` ORDER BY backup_date DESC",
        esc_sql($table_name)
    )
);

// Debug-Information nur bei WP_DEBUG
$backup_count = count($backups);
$show_debug_info = defined('WP_DEBUG') && WP_DEBUG;
$backup_dir_exists = $show_debug_info ? file_exists(JBM_BACKUP_DIR) : true;
$backup_dir_writable = $show_debug_info ? is_writable(JBM_BACKUP_DIR) : true;

// Statistiken
$database = new \JBM\Database();
$file_system = new \JBM\FileSystem();

$db_size = $database->get_database_size();
$plugins_size = $file_system->get_directory_size(WP_CONTENT_DIR . '/plugins');
$themes_size = $file_system->get_directory_size(WP_CONTENT_DIR . '/themes');
$uploads_size = $file_system->get_directory_size(WP_CONTENT_DIR . '/uploads');
$total_size = $db_size + $plugins_size + $themes_size + $uploads_size;

function format_bytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    return round($bytes, $precision) . ' ' . $units[$i];
}
?>

<div class="bmp-container">
    <div class="bmp-header">
        <h1 style="display: flex; align-items: center; gap: 12px;">
            <img src="<?php echo plugins_url('assets/images/jenva-backup-migration-logo.png', dirname(__FILE__)); ?>" alt="Jenva Logo" style="width: 40px; height: 40px; object-fit: contain;">
            <?php _e('Jenva Backup & Migration', 'jenva-backup-migration'); ?>
        </h1>
        <p class="bmp-subtitle"><?php _e('Professionelle Backup-Lösung für WordPress', 'jenva-backup-migration'); ?></p>
    </div>

    <!-- DIAGNOSE-BOX -->
    <?php 
    $settings = get_option('jbm_settings', []);
    $show_diagnose_tool = isset($settings['show_diagnose_tool']) && $settings['show_diagnose_tool'];
    if ($show_diagnose_tool): 
    ?>
    <div class="bmp-card" style="border: 2px solid #ff9500; margin-bottom: 24px;">
        <div class="bmp-card-header" style="background: #fff3cd; border-bottom: 2px solid #ff9500;">
            <h2 style="color: #856404; margin: 0;">
                <?php _e('🔧 Problem: Backups werden nicht angezeigt?', 'jenva-backup-migration'); ?>
            </h2>
        </div>
        <div class="bmp-card-body">
            <p style="color: #856404; margin-bottom: 15px;">
                <?php echo sprintf(__('Dieser Test erstellt ein echtes Backup und zeigt <strong>Schritt für Schritt</strong> was funktioniert und was nicht.', 'jenva-backup-migration')); ?>
            </p>
            <button id="jbm-diagnose-test" class="bmp-button bmp-button-primary" style="margin-bottom: 15px;">
                <?php _e('🔍 Diagnose-Test jetzt starten', 'jenva-backup-migration'); ?>
            </button>
            <div id="jbm-diagnose-output" style="display: none; padding: 20px; background: #1d2327; color: #00ff00; border-radius: 6px; font-family: monospace; font-size: 12px; max-height: 400px; overflow-y: auto; line-height: 1.8;">
                <!-- Output wird hier eingefügt -->
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Statistiken -->
    <div class="bmp-stats-grid">
        <div class="bmp-stat-card">
            <div class="bmp-stat-icon bmp-icon-database">
                <span class="dashicons dashicons-database"></span>
            </div>
            <div class="bmp-stat-content">
                <h3><?php echo format_bytes($db_size); ?></h3>
                <p><?php _e('Datenbank', 'jenva-backup-migration'); ?></p>
            </div>
        </div>
        
        <div class="bmp-stat-card">
            <div class="bmp-stat-icon bmp-icon-plugins">
                <span class="dashicons dashicons-admin-plugins"></span>
            </div>
            <div class="bmp-stat-content">
                <h3><?php echo format_bytes($plugins_size); ?></h3>
                <p><?php _e('Plugins', 'jenva-backup-migration'); ?></p>
            </div>
        </div>
        
        <div class="bmp-stat-card">
            <div class="bmp-stat-icon bmp-icon-themes">
                <span class="dashicons dashicons-admin-appearance"></span>
            </div>
            <div class="bmp-stat-content">
                <h3><?php echo format_bytes($themes_size); ?></h3>
                <p><?php _e('Themes', 'jenva-backup-migration'); ?></p>
            </div>
        </div>
        
        <div class="bmp-stat-card">
            <div class="bmp-stat-icon bmp-icon-uploads">
                <span class="dashicons dashicons-admin-media"></span>
            </div>
            <div class="bmp-stat-content">
                <h3><?php echo format_bytes($uploads_size); ?></h3>
                <p><?php _e('Uploads', 'jenva-backup-migration'); ?></p>
            </div>
        </div>
    </div>

    <!-- Backup erstellen -->
    <div class="bmp-card">
        <div class="bmp-card-header">
            <h2><?php _e('Neues Backup erstellen', 'jenva-backup-migration'); ?></h2>
        </div>
        <div class="bmp-card-body">
            <form id="jbm-create-backup-form" class="bmp-backup-form" method="post" action="javascript:void(0);">
                <div class="bmp-form-row">
                    <div class="bmp-form-group">
                        <label for="backup-name"><?php _e('Backup-Name (optional)', 'jenva-backup-migration'); ?></label>
                        <input type="text" id="backup-name" name="backup_name" class="bmp-input" placeholder="<?php _e('z.B. Vor Update', 'jenva-backup-migration'); ?>">
                    </div>
                </div>
                
                <div class="bmp-form-row">
                    <div class="bmp-form-group">
                        <label><?php _e('Backup-Typ', 'jenva-backup-migration'); ?></label>
                        <div class="bmp-backup-types">
                            <label class="bmp-backup-type-card active">
                                <input type="radio" name="backup_type" value="full" checked>
                                <div class="bmp-type-icon">
                                    <span class="dashicons dashicons-cloud"></span>
                                </div>
                                <h4><?php _e('Vollständig', 'jenva-backup-migration'); ?></h4>
                                <p><?php _e('Datenbank + alle Dateien', 'jenva-backup-migration'); ?></p>
                            </label>
                            
                            <label class="bmp-backup-type-card">
                                <input type="radio" name="backup_type" value="database">
                                <div class="bmp-type-icon">
                                    <span class="dashicons dashicons-database"></span>
                                </div>
                                <h4><?php _e('Datenbank', 'jenva-backup-migration'); ?></h4>
                                <p><?php _e('Nur Datenbank', 'jenva-backup-migration'); ?></p>
                            </label>
                            
                            <label class="bmp-backup-type-card">
                                <input type="radio" name="backup_type" value="files-only">
                                <div class="bmp-type-icon">
                                    <span class="dashicons dashicons-media-document"></span>
                                </div>
                                <h4><?php _e('Dateien', 'jenva-backup-migration'); ?></h4>
                                <p><?php _e('Nur Dateien (ohne DB)', 'jenva-backup-migration'); ?></p>
                            </label>
                            
                            <label class="bmp-backup-type-card">
                                <input type="radio" name="backup_type" value="plugins">
                                <div class="bmp-type-icon">
                                    <span class="dashicons dashicons-admin-plugins"></span>
                                </div>
                                <h4><?php _e('Plugins', 'jenva-backup-migration'); ?></h4>
                                <p><?php _e('Nur Plugin-Dateien', 'jenva-backup-migration'); ?></p>
                            </label>
                            
                            <label class="bmp-backup-type-card">
                                <input type="radio" name="backup_type" value="themes">
                                <div class="bmp-type-icon">
                                    <span class="dashicons dashicons-admin-appearance"></span>
                                </div>
                                <h4><?php _e('Themes', 'jenva-backup-migration'); ?></h4>
                                <p><?php _e('Nur Theme-Dateien', 'jenva-backup-migration'); ?></p>
                            </label>
                            
                            <label class="bmp-backup-type-card">
                                <input type="radio" name="backup_type" value="uploads">
                                <div class="bmp-type-icon">
                                    <span class="dashicons dashicons-admin-media"></span>
                                </div>
                                <h4><?php _e('Uploads', 'jenva-backup-migration'); ?></h4>
                                <p><?php _e('Nur Upload-Dateien', 'jenva-backup-migration'); ?></p>
                            </label>
                        </div>
                    </div>
                </div>
                
                <div class="bmp-form-actions">
                    <button type="submit" class="bmp-button bmp-button-primary bmp-button-large">
                        <span class="dashicons dashicons-update"></span>
                        <?php _e('Backup jetzt erstellen', 'jenva-backup-migration'); ?>
                    </button>
                </div>
            </form>
            
            <div id="jbm-backup-progress" class="bmp-progress-container" style="display: none;">
                <div class="bmp-progress-bar" style="position: relative;">
                    <div class="jbm-progress-bar-fill"></div>
                    <div class="jbm-progress-percentage" style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); font-weight: 700; color: white; text-shadow: 0 1px 2px rgba(0,0,0,0.5); z-index: 10;">
                        0%
                    </div>
                </div>
                <p class="bmp-progress-text"><?php _e('Backup wird erstellt...', 'jenva-backup-migration'); ?></p>
                
                <!-- Motivierende Nachrichten & Tipps während Backup -->
                <div id="jbm-backup-messages" style="margin-top: 20px; padding: 24px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border-radius: 12px; display: none; box-shadow: 0 4px 16px rgba(0,0,0,0.15); min-height: 120px; position: relative; overflow: hidden;">
                    <div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: radial-gradient(circle at top right, rgba(255,255,255,0.1), transparent); pointer-events: none;"></div>
                    <div id="jbm-message-content" style="position: relative; z-index: 1; text-align: center;">
                        <div id="jbm-message-icon" style="font-size: 36px; margin-bottom: 12px;">⏳</div>
                        <div id="jbm-message-text" style="font-size: 16px; font-weight: 500; line-height: 1.6; margin-bottom: 8px;">
                            <?php _e('Dein Backup wird gerade erstellt...', 'jenva-backup-migration'); ?>
                        </div>
                        <div id="jbm-message-tip" style="font-size: 13px; opacity: 0.9; font-style: italic; line-height: 1.5;">
                            <?php _e('💡 Bitte warten...', 'jenva-backup-migration'); ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Backup hochladen -->
    <div class="bmp-card">
        <div class="bmp-card-header">
            <h2><?php _e('Backup hochladen', 'jenva-backup-migration'); ?></h2>
        </div>
        <div class="bmp-card-body">
            <form id="jbm-upload-backup-form" class="bmp-upload-form">
                <div class="bmp-upload-area" id="jbm-upload-area">
                    <span class="dashicons dashicons-upload"></span>
                    <p><?php _e('Ziehen Sie Backup-Dateien hierher oder klicken Sie zum Auswählen', 'jenva-backup-migration'); ?></p>
                    <p style="font-size: 13px; color: #646970; margin: 8px 0 0 0;">
                        <?php _e('💡 Mehrere Dateien gleichzeitig möglich (z.B. für UpdraftPlus)', 'jenva-backup-migration'); ?>
                    </p>
                    <input type="file" id="backup-file" name="backup_files[]" accept=".zip,.wpress,.gz" multiple style="display: none;">
                    <button type="button" class="bmp-button bmp-button-secondary" onclick="document.getElementById('backup-file').click()">
                        <?php _e('Datei(en) auswählen', 'jenva-backup-migration'); ?>
                    </button>
                </div>
                <div id="jbm-selected-files" style="display: none; margin-top: 15px; padding: 15px; background: #f6f7f7; border-radius: 4px;">
                    <h4 style="margin: 0 0 10px 0;"><?php _e('Ausgewählte Dateien:', 'jenva-backup-migration'); ?></h4>
                    <div id="jbm-file-list"></div>
                    <button type="button" id="jbm-upload-start" class="bmp-button bmp-button-primary" style="margin-top: 10px;">
                        <span class="dashicons dashicons-upload"></span>
                        <?php _e('Hochladen starten', 'jenva-backup-migration'); ?>
                    </button>
                </div>
                
                <!-- Batch-Upload Fortschritt -->
                <div id="jbm-batch-progress" style="display: none; margin-top: 15px; padding: 20px; background: #f6f7f7; border-radius: 4px;">
                    <h4 style="margin: 0 0 15px 0;"><?php _e('Upload-Fortschritt', 'jenva-backup-migration'); ?></h4>
                    <div class="bmp-progress-bar" style="height: 12px; background: #e5e5e5; border-radius: 6px; overflow: hidden; margin-bottom: 10px;">
                        <div id="jbm-batch-progress-bar" style="height: 100%; background: linear-gradient(90deg, #2271b1, #72aee6); width: 0%; transition: width 0.3s ease;"></div>
                    </div>
                    <p id="jbm-batch-progress-text" style="margin: 0; text-align: center; font-weight: 600; color: #1d2327;">
                        <?php _e('Vorbereitung...', 'jenva-backup-migration'); ?>
                    </p>
                    <div id="bmp-batch-details" style="margin-top: 10px; font-size: 13px; color: #646970; text-align: center;">
                        <span id="jbm-batch-current">0</span> / <span id="jbm-batch-total">0</span> Batches
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Debug: Letzte Logs anzeigen -->
    <?php
    $logger = new \JBM\Logger();
    $recent_logs = $logger->get_logs(10);
    $has_errors = false;
    foreach ($recent_logs as $log) {
        if (strpos($log, '[error]') !== false) {
            $has_errors = true;
            break;
        }
    }
    
    if ($has_errors && !empty($recent_logs)):
    ?>
        <div class="bmp-card" style="border-left: 4px solid #d63638;">
            <div class="bmp-card-header" style="background: #fef0f0;">
                <h2 style="color: #d63638;">
                    <span class="dashicons dashicons-warning"></span>
                    <?php _e('Letzte Fehler gefunden!', 'jenva-backup-migration'); ?>
                </h2>
            </div>
            <div class="bmp-card-body">
                <div style="background: #1d2327; color: #00ff00; padding: 15px; border-radius: 4px; font-family: monospace; font-size: 12px; max-height: 300px; overflow-y: auto;">
                    <?php foreach ($recent_logs as $log): ?>
                        <?php if (!empty(trim($log))): ?>
                            <div style="<?php echo strpos($log, '[error]') !== false ? 'color: #ff6b6b; font-weight: bold;' : ''; ?>">
                                <?php echo esc_html($log); ?>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
                <p style="margin-top: 15px;">
                    <strong><?php _e('Siehe unten für vollständige Logs:', 'jenva-backup-migration'); ?></strong>
                    <a href="<?php echo admin_url('admin.php?page=backup-migration-settings'); ?>" class="bmp-button bmp-button-secondary bmp-button-small">
                        <?php _e('Zu Einstellungen & Logs', 'jenva-backup-migration'); ?>
                    </a>
                </p>
            </div>
        </div>
    <?php endif; ?>

    <!-- Vorhandene Backups -->
    <div class="bmp-card">
        <div class="bmp-card-header">
            <h2 class="jbm-backups-count-header"><?php _e('Vorhandene Backups', 'jenva-backup-migration'); ?> (<?php echo intval($backup_count); ?>)</h2>
            <button class="bmp-button bmp-button-secondary" id="jbm-refresh-backups">
                <span class="dashicons dashicons-update"></span>
                <?php _e('Aktualisieren', 'jenva-backup-migration'); ?>
            </button>
        </div>
        <div class="bmp-card-body">
            <?php if ($show_debug_info && (!$backup_dir_exists || !$backup_dir_writable)): ?>
                <div class="bmp-alert bmp-alert-warning">
                    <span class="dashicons dashicons-warning"></span>
                    <div>
                        <strong><?php _e('Hinweis:', 'jenva-backup-migration'); ?></strong>
                        <?php if (!$backup_dir_exists): ?>
                            <?php _e('Backup-Verzeichnis existiert nicht. Es wird beim ersten Backup erstellt.', 'jenva-backup-migration'); ?>
                        <?php elseif (!$backup_dir_writable): ?>
                            <?php _e('Backup-Verzeichnis ist nicht beschreibbar. Bitte Berechtigungen überprüfen:', 'jenva-backup-migration'); ?>
                            <code><?php echo esc_html(JBM_BACKUP_DIR); ?></code>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
            <div id="jbm-backups-list">
            <?php if (empty($backups)): ?>
                <div class="bmp-empty-state">
                    <span class="dashicons dashicons-cloud"></span>
                    <h3><?php _e('Keine Backups vorhanden', 'jenva-backup-migration'); ?></h3>
                    <p><?php _e('Erstellen Sie Ihr erstes Backup, um zu beginnen.', 'jenva-backup-migration'); ?></p>
                </div>
            <?php else: ?>
                    <?php foreach ($backups as $backup): 
                        $is_protected = !empty($backup->protected);
                    ?>
                        <div class="bmp-backup-item <?php echo $is_protected ? 'bmp-backup-protected' : ''; ?>" data-backup-id="<?php echo intval($backup->id); ?>" data-protected="<?php echo $is_protected ? '1' : '0'; ?>">
                            <div class="bmp-backup-icon">
                                <span class="dashicons dashicons-<?php echo $is_protected ? 'lock' : 'archive'; ?>" style="<?php echo $is_protected ? 'color: #d63638;' : ''; ?>"></span>
                            </div>
                            <div class="bmp-backup-info">
                                <h4>
                                    <?php echo esc_html($backup->backup_name); ?>
                                    <?php if ($is_protected): ?>
                                        <span class="jbm-protected-badge" style="display: inline-block; margin-left: 8px; padding: 2px 8px; background: #d63638; color: white; border-radius: 3px; font-size: 11px; font-weight: 600;">
                                            🔒 <?php _e('GESCHÜTZT', 'jenva-backup-migration'); ?>
                                        </span>
                                    <?php endif; ?>
                                </h4>
                                <div class="bmp-backup-meta">
                                    <span class="bmp-backup-type">
                                        <span class="dashicons dashicons-tag"></span>
                                        <?php echo esc_html($backup->backup_type ?? ''); ?>
                                    </span>
                                    <span class="bmp-backup-size">
                                        <span class="dashicons dashicons-chart-bar"></span>
                                        <?php echo esc_html(format_bytes($backup->backup_size ?? 0)); ?>
                                    </span>
                                    <span class="bmp-backup-date">
                                        <span class="dashicons dashicons-calendar"></span>
                                        <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($backup->backup_date ?? 'now'))); ?>
                                    </span>
                                </div>
                            </div>
                            <div class="bmp-backup-actions">
                                <button class="bmp-button bmp-button-small <?php echo $is_protected ? 'bmp-button-warning' : 'bmp-button-secondary'; ?> jbm-toggle-protection" data-backup-id="<?php echo intval($backup->id); ?>" title="<?php echo $is_protected ? esc_attr__('Schutz aufheben', 'jenva-backup-migration') : esc_attr__('Vor Auto-Löschung schützen', 'jenva-backup-migration'); ?>">
                                    <span class="dashicons dashicons-<?php echo $is_protected ? 'unlock' : 'lock'; ?>"></span>
                                    <span class="protection-text"><?php echo $is_protected ? __('Entsperren', 'jenva-backup-migration') : __('Schützen', 'jenva-backup-migration'); ?></span>
                                </button>
                                <button class="bmp-button bmp-button-small bmp-button-success jbm-restore-backup" data-backup-id="<?php echo intval($backup->id); ?>">
                                    <span class="dashicons dashicons-backup"></span>
                                    <?php _e('Wiederherstellen', 'jenva-backup-migration'); ?>
                                </button>
                                <a href="<?php echo esc_url(admin_url('admin-ajax.php?action=jbm_download_backup&backup_id=' . intval($backup->id ?? 0) . '&nonce=' . wp_create_nonce('jbm_nonce'))); ?>" class="bmp-button bmp-button-small bmp-button-secondary jbm-download-backup" data-backup-id="<?php echo intval($backup->id ?? 0); ?>" data-backup-size="<?php echo intval($backup->backup_size ?? 0); ?>">
                                    <span class="dashicons dashicons-download"></span>
                                    <?php _e('Herunterladen', 'jenva-backup-migration'); ?>
                                </a>
                                <button class="bmp-button bmp-button-small bmp-button-danger jbm-delete-backup" data-backup-id="<?php echo intval($backup->id); ?>" <?php echo $is_protected ? 'disabled title="' . esc_attr__('Geschützte Backups können nicht gelöscht werden', 'jenva-backup-migration') . '"' : ''; ?>>
                                    <span class="dashicons dashicons-trash"></span>
                                    <?php _e('Löschen', 'jenva-backup-migration'); ?>
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
            <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div id="jbm-delete-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0, 0, 0, 0.7); z-index: 100000; align-items: center; justify-content: center;">
    <div style="background: white; border-radius: 12px; padding: 30px; max-width: 500px; width: 90%; box-shadow: 0 8px 32px rgba(0,0,0,0.3); position: relative;">
        <div style="text-align: center; margin-bottom: 20px;">
            <div style="font-size: 48px; margin-bottom: 50px;">🗑️</div>
            <h2 id="jbm-delete-modal-title" style="margin: 0 0 15px 0; color: #d63638; font-size: 22px;"></h2>
        </div>
        <div style="margin-bottom: 20px;">
            <p id="jbm-delete-modal-message" style="margin: 0 0 15px 0; font-size: 15px; line-height: 1.6; color: #1d2327;"></p>
            <div id="jbm-delete-modal-warning" style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 12px 15px; border-radius: 4px; margin-top: 15px;">
                <p style="margin: 0; font-size: 14px; color: #856404; font-weight: 600;"></p>
            </div>
        </div>
        <div style="display: flex; gap: 10px; justify-content: flex-end;">
            <button id="jbm-delete-modal-cancel" class="bmp-button bmp-button-secondary" style="min-width: 120px;">
                <span class="dashicons dashicons-no"></span>
                <span class="cancel-text"></span>
            </button>
            <button id="jbm-delete-modal-confirm" class="bmp-button bmp-button-danger" style="min-width: 120px;">
                <span class="dashicons dashicons-trash"></span>
                <span class="confirm-text"></span>
            </button>
        </div>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    // Diagnose-Test Button
    $('#jbm-diagnose-test').on('click', function() {
        const button = $(this);
        const output = $('#jbm-diagnose-output');
        
        button.prop('disabled', true).text(jbmAdmin.strings.diagnose_test_executing);
        output.show().html('<div style="color: #ffc107;">' + jbmAdmin.strings.diagnose_test_running + '</div>');
        
        $.ajax({
            url: jbmAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'jbm_diagnose_backup',
                nonce: jbmAdmin.nonce
            },
            success: function(response) {
                console.log('🔍 DIAGNOSE-RESPONSE:', response);
                
                if (response.success) {
                    const d = response.data;
                    const s = jbmAdmin.strings;
                    let html = '<div style="font-size: 13px;">';
                    html += '<strong style="color: #ffc107; font-size: 15px;">' + s.diagnose_result + '</strong><br><br>';
                    
                    // Backup erstellt?
                    html += (d.backup_created ? '<span style="color: #00ff00;">✅</span>' : '<span style="color: #ff6b6b;">❌</span>');
                    html += ' ' + s.diagnose_backup_create + (d.backup_created ? s.diagnose_success : s.diagnose_failed) + '<br>';
                    html += '   ' + s.diagnose_backup_name + d.backup_name + '<br><br>';
                    
                    // Datei vorhanden?
                    html += (d.file_exists ? '<span style="color: #00ff00;">✅</span>' : '<span style="color: #ff6b6b;">❌</span>');
                    html += ' ' + s.diagnose_zip_file + (d.file_exists ? s.diagnose_exists : s.diagnose_not_exists) + '<br>';
                    html += '   ' + s.diagnose_path + d.file_path + '<br>';
                    html += '   ' + s.diagnose_size + d.file_size + '<br><br>';
                    
                    // DB-Eintrag?
                    html += (d.db_entry_exists ? '<span style="color: #00ff00;">✅</span>' : '<span style="color: #ff6b6b;">❌</span>');
                    html += ' ' + s.diagnose_db_entry + (d.db_entry_exists ? s.diagnose_present + d.db_id + ')' : s.diagnose_not_found) + '<br><br>';
                    
                    html += '<strong style="color: #ffc107;">' + s.diagnose_summary + '</strong><br>';
                    html += s.diagnose_total_backups + d.total_backups + '<br>';
                    html += s.diagnose_dir_writable + (d.dir_writable ? s.diagnose_yes : '<span style="color: #ff6b6b;">' + s.diagnose_no + '</span>') + '<br><br>';
                    
                    // Letzte Backups
                    if (d.recent_backups && d.recent_backups.length > 0) {
                        html += '<strong style="color: #ffc107;">' + s.diagnose_recent_backups + '</strong><br>';
                        d.recent_backups.forEach(function(bk) {
                            html += '  • ID ' + bk.id + ': ' + bk.backup_name + ' (' + Math.round(bk.backup_size / 1024 / 1024 * 100) / 100 + ' MB)<br>';
                        });
                    }
                    
                    html += '</div>';
                    
                    // Komplette Daten
                    html += '<details style="margin-top: 15px;"><summary style="cursor: pointer; color: #ffc107; font-weight: bold;">' + s.diagnose_show_full_data + '</summary>';
                    html += '<pre style="color: #00ff00; font-size: 10px; margin-top: 10px; white-space: pre-wrap;">' + JSON.stringify(d, null, 2) + '</pre>';
                    html += '</details>';
                    
                    output.html(html);
                    
                    // Wenn erfolgreich, Erfolgsmeldung anzeigen (aber Log geöffnet lassen)
                    if (d.file_exists && d.db_entry_exists) {
                        html += '<br><br><div style="color: #00ff00; font-size: 14px; text-align: center; font-weight: bold;">' + s.diagnose_all_works + '</div>';
                        output.html(html);
                        
                        // Log bleibt geöffnet - nur bei Refresh wird es geschlossen
                        // Backup-Liste im Hintergrund aktualisieren (ohne Reload)
                        setTimeout(function() {
                            // Backup-Liste aktualisieren ohne Reload
                            if (typeof JBM !== 'undefined' && typeof JBM.refreshBackups === 'function') {
                                JBM.refreshBackups();
                            }
                        }, 1000);
                    }
                    
                } else {
                    output.html('<div style="color: #ff6b6b;">' + jbmAdmin.strings.diagnose_error + (response.data.message || 'Unbekannt') + '</div>');
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX-Fehler:', xhr, status, error);
                output.html('<div style="color: #ff6b6b;">' + jbmAdmin.strings.diagnose_ajax_error_title + '<br>' + error + '<br>Status: ' + status + '<br><br>' + jbmAdmin.strings.diagnose_check_console + '</div>');
            },
            complete: function() {
                button.prop('disabled', false).text('🔍 Diagnose-Test jetzt starten');
            }
        });
    });
});
</script>

